
/* Copyright (C) 2001-2010 Monotype Imaging Inc. All rights reserved. */

/* Confidential Information of Monotype Imaging Inc. */

/* fs_graymap.c */

#include "fs_itype.h"
#include "fs_graymap.h"
#include "fs_effects.h"
#include "fs_scratch.h"

#ifdef FS_GRAYMAPS


#define ONE 65536L


/****************************************************************/
/* private support functions for grayscale rasterizer           */
/****************************************************************/
/* compute the length of the control polygon of the character   */
/* it is not less than the actual arc length, and gets closer   */
/* to the actual arc length, with denser points on curves.      */
static int arc_length(FS_OUTLINE *outl)
{
    FS_FIXED *x = (FS_FIXED *)(outl->x);
    FS_FIXED *y = (FS_FIXED *)(outl->y);
    int i, len, num = outl->num;
    FS_BYTE *type = (FS_BYTE *)(outl->type);
    FS_FIXED x0, y0, x1, y1, dx, dy;

    for (x0 = 0, y0 = 0, len = 0, i = 0; i < num; i++)
    {
        switch (type[i])
        {
        case FS_CUBETO:
            x1 = *x++;
            y1 = *y++;
            dx = x1 - x0;
            if (dx < 0) dx = -dx;
            dy = y1 - y0;
            if (dy < 0) dy = -dy;
            len += dx + dy;
            x0 = x1;
            y0 = y1;
            /* FALLTHROUGH */
        case FS_QUADTO:
            x1 = *x++;
            y1 = *y++;
            dx = x1 - x0;
            if (dx < 0) dx = -dx;
            dy = y1 - y0;
            if (dy < 0) dy = -dy;
            len += dx + dy;
            x0 = x1;
            y0 = y1;
            /* FALLTHROUGH */
        case FS_LINETO:
            x1 = *x++;
            y1 = *y++;
            dx = x1 - x0;
            if (dx < 0) dx = -dx;
            dy = y1 - y0;
            if (dy < 0) dy = -dy;
            len += dx + dy;
            x0 = x1;
            y0 = y1;
            break;
        case FS_MOVETO:
        case FS_MOVETO | OUTLINE_CHAR:
            x0 = *x++;
            y0 = *y++;
            break;
        default:
            break;
        }
    }
    return len >> 16;
}

/****************************************************************/
static int add_thang(RASTER *r, FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1)
{
    int y, row;
    FS_FIXED h, m, w, a;
    THANG *thang;

    if (r->next_thang >= NUM_THANGS)
        return 0;

    y = (y0 + y1) >> 17;  /* trunc(average(y0,y1)) */

    row = r->top - y;
    if (row < 0 || row >= NUM_ROWS)
        return 1;    /* it was successful ... in a strange way */

    h = y1 - y0;
    m = (x0 + x1) >> 1;
    w = ONE - (m - FIXED_FLOOR(m));

    if (h >= ONE)
        a = w;
    else if (w >= ONE)
        a = h;
    else
    {
        /* a = FixMul(w,h) -- but we can do it in 32 bits since w<ONE and h<ONE */
        FS_ULONG z = w;
        z *= h;
        z += 32768;
        a = (FS_FIXED)(z >> 16);
    }

    /* get and fill the thang */
    thang = r->thangs + r->next_thang++;
    thang->x = (FS_SHORT)(m >> 16);
    thang->y = (FS_SHORT)y;


    if (r->dir > 0)
    {
        thang->a = a;
        thang->f = h;
    }
    else
    {
        thang->a = -a;
        thang->f = -h;
    }

    /* prepend this thang to the proper row of thangs */
    thang->next = r->rows[row];
    r->rows[row] = thang;

    return 1;
}

#define FS_SLOPE_MAX (1<<30)
#define FS_SLOPE_MIN -FS_SLOPE_MAX

/****************************************************************/
/* update the THANGs the line interacts with                    */
static int do_line(RASTER *r, FS_FIXED x0, FS_FIXED y0, FS_FIXED x1, FS_FIXED y1)
{
    int ok;
    FS_FIXED dx, dy, z;
    FS_FIXED ix0, ix1, iy0, iy1;

    /* ignore horizontal lines */
    if (y0 == y1)
        return 1;

    /* make the line point upwards */
    if (y0 > y1)
    {
        r->dir = -1;    /* but, remember it was pointing down */
        z = x0;
        x0 = x1;
        x1 = z;
        z = y0;
        y0 = y1;
        y1 = z;
    }
    else
        r->dir = 1;


    dx = x1 - x0;
    dy = y1 - y0;

    /* find the starting and ending pixels for x and y.  */
    /* for this purpose, an integer ending pixel needs   */
    /* to be in the previous pixel ... for example       */
    /* x0==8 and x1==9 ... it's all in pixel 8 = [8..9]  */
    iy0 = FIXED_FLOOR(y0);
    iy1 = FIXED_CEILING(y1) - ONE;
    if (dx == 0)
        ix0 = ix1 = FIXED_FLOOR(x0);
    else if (dx > 0)
    {
        ix0 = FIXED_FLOOR(x0);
        ix1 = FIXED_CEILING(x1) - ONE;
    }
    else
    {
        ix1 = FIXED_FLOOR(x1);
        ix0 = FIXED_CEILING(x0) - ONE;
    }

    /*
    ** first, some quicker special cases
    */

    /* line lives in a single pixel */
    if (ix0 == ix1 && iy0 == iy1)
    {
        ok = add_thang(r, x0, y0, x1, y1);
        return ok;
    }

    /* line is vertical */
    if (x0 == x1)
    {
        FS_FIXED y;

        if (r->num_bands > 1)
        {
            /* limit the line to the current band */
            if (y0 < r->band_ymin)
                y0 = r->band_ymin;
            if (y1 > r->band_ymax)
                y1 = r->band_ymax;
        }

        y = FIXED_FLOOR(y0) + ONE;
        if (y > y1) y = y1;
        ok = add_thang(r, x0, y0, x0, y);
        if (!ok) return 0;

        while (y + ONE < y1)
        {
            ok = add_thang(r, x0, y, x0, y + ONE);
            if (!ok) return 0;
            y += ONE;
        }

        if (y < y1)
            ok = add_thang(r, x0, y, x0, y1);

        return ok;
    }


    /* line is nearly vertical (lives in a single column of pixels) */
    if (ix0 == ix1)
    {
        FS_FIXED x, y, recip;

        recip = FixDiv(dx, dy);

        /* limit the line to the current band */
        if (r->num_bands > 1)
        {
            if (y0 < r->band_ymin)
            {
                y0 = r->band_ymin;
                x0 = x1 - FixMul(y1 - y0, recip);
            }

            if (y1 > r->band_ymax)
            {
                y1 = r->band_ymax;
                x1 = x0 + FixMul(y1 - y0, recip);
            }
        }

        y = FIXED_FLOOR(y0) + ONE;
        if (y > y1)
        {
            x = x1; /* line ends inside pixel */
            y = y1;
        }
        else
            x = x0 + FixMul(y - y0, recip);

        ok = add_thang(r, x0, y0, x, y);
        if (!ok) return 0;

        while ((y + ONE) < y1)
        {
            ok = add_thang(r, x, y, x + recip, y + ONE);
            if (!ok) return 0;
            x += recip;
            y += ONE;
        }

        if (y < y1)
            ok = add_thang(r, x, y, x1, y1);

        return ok;
    }

    /* line is nearly horizontal (lives in a single row of pixels) */
    if (iy0 == iy1)
    {
        FS_FIXED x, y, slope;

        slope = FixDiv(dy, dx);
        if (dx > 0)
        {
            x = FIXED_FLOOR(x0) + ONE; /* next integer above x0 */
            y = y0 + FixMul(x - x0, slope);
            ok = add_thang(r, x0, y0, x, y);

            while ((x + ONE) < x1)
            {
                ok = add_thang(r, x, y, x + ONE, y + slope);
                if (!ok) return 0;
                x += ONE;
                y += slope;
            }

            if (x < x1)
                ok = add_thang(r, x, y, x1, y1);
            return ok;
        }
        else
        {
            x = FIXED_CEILING(x0) - ONE; /* next integer below x0 */
            y = y0 + FixMul(x - x0, slope);
            ok = add_thang(r, x0, y0, x, y);

            while ((x - ONE) > x1)
            {
                ok = add_thang(r, x, y, x - ONE, y - slope);
                if (!ok) return 0;
                x -= ONE;
                y -= slope;
            }

            if (x > x1)
                ok = add_thang(r, x, y, x1, y1);
            return ok;
        }
    }

    /*
    ** ... the generic line
    */
    {
        /* we are interested in every place where the x or y
        ** coordinate is integer along this line -- and of
        ** course the end-points
        **                                             (x1,y1)
        **      + - - - - - + - - - - - + - - - - - + - - @ - - +
        **      |           |           |           |   o       |
        **      |           |           |           | o         |
        **      |           |           |           @           |
        **      |           |           |         o |           |
        **      |           |           |       o   |           |
        **      |           |           |     o     |           |
        **      + - - - - - + - - - - - + - @ - - - + - - - - - +
        **      |           |           | o         |           |
        **      |           |           @           |           |
        **      |           |         o |           |           |
        **      |           |       o   |           |           |
        **      |           |     o     |           |           |
        **      |           |   o       |           |           |
        **      + - - - - - + @ - - - - + - - - - - + - - - - - +
        **      |           @           |           |           |
        **      |         o |           |           |           |
        **      |       @   |           |           |           |
        **      |    (x0,y0)|           |           |           |
        **      |           |           |           |           |
        **      |           |           |           |           |
        **      + - - - - - + - - - - - + - - - - - + - - - - - +
        */

        FS_FIXED slope, recip;
        FS_FIXED iy, xiy, ix, yix, d_ix, d_slope;

        slope = FixDiv(dy, dx);

        /* handle cases where slope is very nearly horizontal or vertical */
        if (slope == 0) return 1; /* very nearly horizontal line */
        if (slope > FS_SLOPE_MAX || slope < FS_SLOPE_MIN) /* very nearly vertical line */
            slope >>= 1; /* avoid fixed point overflow for nearly vertical lines */

        recip = FixDiv(dx, dy);

        /* limit the line to the current band */
        if (r->num_bands > 1)
        {
            if (y0 < r->band_ymin)
            {
                y0 = r->band_ymin;
                x0 = x1 - FixMul(y1 - y0, recip);
            }

            if (y1 > r->band_ymax)
            {
                y1 = r->band_ymax;
                x1 = x0 + FixMul(y1 - y0, recip);
            }
        }

        /* next integer y and corresponding x .. think xiy = x(iy) */
        iy = FIXED_FLOOR(y0) + ONE;
        xiy = x0 + FixMul(iy - y0, recip);

        /* next integer x and corresponding y .. think yix == y(ix) */
        if (dx > 0)
        {
            d_ix = ONE;
            d_slope = slope;
            ix = FIXED_FLOOR(x0) + ONE;
        }
        else
        {
            d_ix = -ONE;
            d_slope = -slope;
            ix = FIXED_CEILING(x0) - ONE;
        }
        yix = y0 + FixMul(ix - x0, slope);

        /* we now have two ways of stepping up the line,       */
        /* by integer y's OR by integer x's ... at each        */
        /* iteration we use the one that has the lower y value */
        /* we bump the one we used, to the next position.      */
        /* thus, the two points play "leap-frog" up the line   */
        for (;;)
        {
            if (iy > y1 && yix > y1)
            {
                /* ? both too high, we're done */
                if (x0 != x1 || y0 != y1)
                {
                    ok = add_thang(r, x0, y0, x1, y1);
                    if (!ok) return 0;
                }
                break;
            }

            if (iy < yix)
            {
                ok = add_thang(r, x0, y0, xiy, iy);
                if (!ok) return 0;
                y0 = iy;
                x0 = xiy;
                iy += ONE;
                xiy += recip;
            }
            else
            {
                ok = add_thang(r, x0, y0, ix, yix);
                if (!ok) return 0;
                x0 = ix;
                y0 = yix;
                ix += d_ix;
                yix += d_slope;
            }
        }
    }
    return 1;
}

/****************************************************************/
static void do_band(RASTER *r)
{
    THANG *thang, **rows = r->rows;
    FS_FIXED  *areas = r->areas;
    FS_FIXED  *nexts = r->nexts;
    FS_GRAYMAP *gmap = r->gmap;
    int i, width = gmap->width;
    int max = r->band_ymax >> 16;
    int min = r->band_ymin >> 16;
    int n = 1 + max - min;
    int row = gmap->hi_y - max + 1;  /* first row not used */
    FS_BYTE *grays = gmap->bits + gmap->bpl * row;

    /* not sure why this would happen, but... */
    if (n < 0 || n >= NUM_ROWS)
        return;

    if (areas == 0)
        return; /* critcally low memory situation */

    /* process the thangs associated with each row of the band */
    for (i = 1; i < n; i++)
    {
        FS_LONG first = width;
        FS_LONG last  = 0;
        /* remember: areas[] and nexts[] were created */
        /* with a single malloc of width * 8 bytes.   */
        /* this zeros both areas[] and nexts[]        */
        SYS_MEMSET(areas, 0, width << 3);

        for (thang = rows[i]; thang; thang = thang->next)
        {
            register int c = thang->x - gmap->lo_x;

            /* this should never happen ... but */
            if (c < 0)
                c = 0;

            /* Keep track of highest and lowest position, so we can avoid */
            /* iterating over blank space.                                */
            if (c < first)
                first = c;

            if (c > last)
                last = c;

            /* must use "+=" since multiple thangs might have same thang->x  */
            /* c could be equal to width for gridfitted outlines, this is ok */
            if (c < width)
            {
                areas[c] += thang->a;
                nexts[c] += thang->f;
            }
        }

        /* now process areas[] and next[] to 8-bit grays     */
        /* Use a new block with some hints for the compiler  */
        {
            register FS_FIXED a;
            register int j;
            FS_FIXED next = 0;

            /* Last can be greater than width, so check against width also. */
            for (j = first; (j <= last) && (j < width); j++)
            {
                a = next + areas[j];
                if (a)
                {
                    /* correct sign */
                    if (a < 0) a = -a;
                    /* scale */
                    a >>= 8;
                    /* limit */
                    if (a > 255) a = 255;
                    /* assign */
                    grays[j] = (FS_BYTE)a;
                }
                next += nexts[j];
            }
        }

        grays += width;
    }
}

/****************************************************************/
/* create 8-bit graymap                                         */
static FS_GRAYMAP *new_graymap_8(_DS_ FS_OUTLINE *outl)
{
    FS_GRAYMAP *gmap;
    int size, height, width;
    int xmin, ymin, xmax, ymax;

    xmin = FIXED_FLOOR(outl->lo_x) >> 16;
    ymin = FIXED_FLOOR(outl->lo_y) >> 16;
    xmax = FIXED_CEILING(outl->hi_x) >> 16;
    ymax = FIXED_CEILING(outl->hi_y) >> 16;

#ifdef FS_PSEUDO_BOLD
    if   ( (STATE.flags & FLAGS_ADD_WEIGHT) ||
           ((STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) &&
            (STATE.lpm <= 26) && !(STATE.flags & FLAGS_OUTLINEBOLD)) )
    {
        SENV *senv = (SENV *)(STATE.cur_sfnt->senv);
        FS_SHORT bw = senv->bold_width;
        if (bw)
            xmin--;
        if (bw > 1)
            xmax++;
    }
#endif

    height = ymax - ymin; /* to avoid trimming empty first row */
    width  = xmax - xmin;

    /* allocate it */
    size = sizeof(FS_GRAYMAP) - 1;
    size += width * height;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "NEW_GRAYMAP_8";
#endif
    gmap = FSS_malloc(_PS_ size);

    if (gmap)
    {
        /* initialize it */
        SYS_MEMSET(gmap, 0, size);
        gmap->size = size;
        gmap->lo_x = (FS_SHORT)xmin;
        gmap->hi_y = (FS_SHORT)ymax - 1; /* to avoid trimming empty first row */
        gmap->width = (FS_SHORT)width;
        gmap->height = (FS_SHORT)height;
        gmap->bpl = (FS_SHORT)width;
        gmap->bitsPerPixel = 8;
        gmap->dx = outl->dx;
        gmap->dy = outl->dy;
        gmap->i_dx = outl->i_dx;
        gmap->i_dy = outl->i_dy;
        gmap->type = FS_MAP_GRAYMAP8;
        gmap->embedded = false;        /* this graymap was outline-generated */
    }

    return gmap;
}

/****************************************************************/
/* make_graymap - algorithm description
**
** Convert outline into line segments and create THANGs for any
** any line segment crossing raster line @ y -- i.e. in the interval [y .. y+1]
**
**                 x1,y1
**    + - - - - - - -+- - + - - - - - - - - - +
**    |             /#####|###################|
**    |            /######|###################|
**    |           /## a ##|######## f ########|   THANG data:
**    |          /########|###################|   a = area that is covered in this pixel == w * h ... w=(x0+x1)/2, h=(y1-y0)
**    |         /#########|###################|   f = area that is covered in subsequent pixels == one * h == h
**    |        /##########|###################|
**    |     x0,y0         |                   |      both a and f are SIGNED, '+' for on transitions, '-' for off transitions
**    |                   |                   |
**    |                   |                   |    0 <= a <= one
**    |                   |                   |       0 <= f <= one
**    + - - - - - - - - - + - - - - - - - - - +
**  (x,y)
**
**  For each raster line (row) convert areas and nexts (f) into gray values
**
**  To convert outline to lines:
**  Use an internal  stack for quads/cubes.  If the curve on top of stack isn't
**  flat enough, replace it with the half-curves split at the parametric midpoint.
**
**  Sooner or later the curve on the top of the stack will be flat enough and we
**  can consider it to be a line.
**
**  if maximum subdivision is level 4 ... need at most 5 quads on the stack
**  000000
**  111111 111111
**  111111 222222 222222
**  111111 222222 333333 333333
**  111111 222222 333333 444444 444444
**  111111 222222 333333 444444
**  111111 222222 333333
**  111111 222222 444444 444444
**  111111 222222 444444
**  111111 222222
**  111111 333333 333333
**  111111 333333 444444 444444
**  111111 333333 444444
**  111111 333333
**  111111 444444 444444
**  111111 444444
**  111111
**  222222 222222
**  222222 333333 333333
**  222222 333333 444444 444444
**  222222 333333 444444
**  222222 333333
**  222222 444444 444444
**  222222 444444
**  222222
**  333333 333333
**  333333 444444 444444
**  333333 444444
**  333333
**  444444 444444
**  444444
*/

/****************************************************************/
FS_GRAYMAP *make_graymap(_DS_ FS_OUTLINE *outl, FS_USHORT gmaptype)
{
    FS_USHORT outline_width = STATE.outline_width;
    FS_FIXED opacity = STATE.outline_opacity;
    FS_LONG size;
    FS_GRAYMAP *gmap = 0;
    FS_SHORT bpp = 4;
    FS_BOOLEAN needs_trim_convert = 1;

#ifdef FS_PSEUDO_BOLD
    FS_BOOLEAN pixel_bold = 0;
#endif /* FS_PSEUDO_BOLD */

#ifdef FS_STIK
    FS_LONG n_types, n_points;
    int is_stik = 0;
#endif

    if (outl == 0)
        return 0;

    if (gmaptype & FS_MAP_GRAYMAP4)
    {
        bpp = 4;
        gmaptype = FS_MAP_GRAYMAP4;
    }
    else if (gmaptype & FS_MAP_GRAYMAP8)
    {
        bpp = 8;
        gmaptype = FS_MAP_GRAYMAP8;
    }
    else if (gmaptype & FS_MAP_GRAYMAP2)
    {
        bpp = 2;
        gmaptype = FS_MAP_GRAYMAP2;
    }

    /* space character */
    if (outl->num == 0)
    {
        size = offsetof(FS_GRAYMAP, bits);
#ifdef FS_MEM_DBG
        STATE.memdbgid = "FS_GRAYMAP";
#endif
        gmap = (FS_GRAYMAP *) FSS_calloc(_PS_ size);
        if (gmap == 0)
            return 0;
        gmap->size = size;
        gmap->dx = outl->dx;
        gmap->dy = outl->dy;
        gmap->i_dx = outl->i_dx;
        gmap->i_dy = outl->i_dy;
        gmap->embedded = false;        /* this graymap was outline-generated */
        gmap->bitsPerPixel = bpp;
        gmap->type = gmaptype; 
#ifdef FS_STATS
        made_gmap++;
#endif
        return gmap; 
    }

#ifdef FS_STIK 
    /* for graymaps we always expand the stick ... no direct drawing */
    if ((STATE.cur_lfnt->fontflags & FONTFLAG_STIK) && !(outl->type[0] & OUTLINE_CHAR))
    {
        is_stik = 1;                    /* we have a stik */
        /* outl passed in will be deleted by caller of make_graymap */
        /* outl returned will be deleted by "if (is_stik)" below */
        outl = expand_stik(_PS_ outl, &n_types, &n_points);
        if (!outl)
        {
            return 0;
        }

#ifdef FS_PSEUDO_BOLD
        if (STATE.bold_pct || STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct)
        {
            if ( (STATE.flags & FLAGS_OUTLINEBOLD) ||
                 ( !(STATE.flags & FLAGS_ADD_WEIGHT) &&
                   ( !( STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) || STATE.lpm > 26)) )
            {
                FS_OUTLINE *tmp;
                tmp = embolden_outline(_PS_ outl, &n_types, &n_points);
                if (!tmp)
                {
#ifdef FS_STIK
                    if (is_stik)
                        FSS_free_char(_PS_ outl);
#endif
                    return 0;
                }

                /* embolden_outline MAY return the original <outl> ... don't free it */
                if (tmp != outl)
                    FSS_free_char(_PS_ outl);
                outl = tmp;
            }
        }
#endif /* FS_PSEUDO_BOLD */
        if (STATE.error)
        {
#ifdef FS_STIK
            if (is_stik)
                FSS_free_char(_PS_ outl);
#endif
            return 0;
        }
    } /* if needed to call expand_stik */
#endif /* FS_STIK */


    /* graymap rendering  */
    /*lint -e801  Info -- Use of goto is deprecated */
    {
        RASTER *r;
        FS_FIXED x0, y0, x1, y1, x2, y2;
        FS_FIXED ymin, ymax;
        FS_FIXED *x, *y;
        FS_BYTE *type = (FS_BYTE *)(outl->type);
        int i, num = outl->num;
        int band, num_bands;
        FS_FIXED band_ymin, band_ymax;
        FS_FIXED *stack;
        FS_FIXED *stack_base;
        int ok;
        FS_FIXED err, max_err = STATE.lpm << 7; /* lpm/512 */
        FS_BOOLEAN bands_increased = 0;

        if (STATE.lpm > 64)
            max_err = 8192;  /* 1/8 pixel */
        else if (STATE.lpm < 16)
            max_err = 2048;  /* 1/32 pixel */

        if (STATE.server->raster == 0)
        {
#ifdef FS_MEM_DBG
            STATE.memdbgid = "RASTER";
#endif
            STATE.server->raster = FSS_malloc(_PS_ sizeof(RASTER));    /* this is big ~ 9K, use workspace */
            if (STATE.server->raster == 0 || STATE.error)
            {
                return 0;
            }
            STATE.server->raster->areas = 0;
            STATE.server->raster->areasize = 50;  /* preallocate now to reduce fragmentation later */
#ifdef FS_MEM_DBG
            STATE.memdbgid = "RASTER areas";
#endif
            STATE.server->raster->areas = (FS_FIXED *)FSS_malloc(_PS_ 2 * STATE.server->raster->areasize * sizeof(FS_FIXED));
            if (STATE.server->raster->areas == 0)
            {
                STATE.server->raster->areasize = 0; /* force allocation next time */
            }

            STATE.server->raster->ref_count = 0;
        }
        r = STATE.server->raster;
        r->ref_count++; /* prevent get_some_back from recovering RASTER */

        /* allocate room for the graymap */
        r->gmap = new_graymap_8(_PS_ outl);
        if (r->gmap == 0)
        {
            r->ref_count--;
            return 0;
        }

        stack_base = r->stack_base;

        r->next_thang = 0;

        /* also need two FIXED temps the width of the graymap */
        /* use one malloc for both                            */
        i = r->gmap->width;
        if (r->areasize < i)
        {
            if (r->areasize)
                FSS_free(_PS_ r->areas);
            r->areasize = i;
#ifdef FS_MEM_DBG
            STATE.memdbgid = "RASTER areas";
#endif
            r->areas = (FS_FIXED *)FSS_malloc(_PS_ 2 * i * sizeof(FS_FIXED));
            if (r->areas == 0)
            {
                r->areasize = 0; /* force allocation next time */
                r->ref_count--;
                FSS_free_char(_PS_ r->gmap);
                return 0;
            }
        }
        r->nexts = r->areas + i;

        /* guess the outline will need a THANG for every       */
        /* pixel of arc length -- using the manhattan metric.  */
        /* So the number of bands is                           */
        r->num_bands = 1 + arc_length(outl) / NUM_THANGS;

        /* if we guessed wrong about the number of bands       */
        /* we will need to start over -- from the beginning    */
        /* with more bands                                     */
restart:
        if ((r->num_bands > r->gmap->height) && bands_increased) /* we have run out of thangs in one row */
        {
            if (max_err > 32768) /* increasing max_err was not effective, bail out */
            {
                STATE.error = ERR_RASTER_RESOURCE_LIMIT;
                return 0;
            }
            else
                max_err += 8192; /* try increasing max_err to reduce # of thangs needed */
        }
        else
        {
            if (r->next_thang > 0)
            {
                r->num_bands <<= 1;
                bands_increased = 1;
            }
        }

        stack = r->stack_base;
        num_bands = r->num_bands;

        for (band = 0; band < num_bands; band++)
        {
            /* each band starts from scratch, so clear the  */
            /* linked lists for all rows                    */
            SYS_MEMSET(r->rows, 0, NUM_ROWS * sizeof (THANG *));

            /* each band uses all the available thangs. */
            r->next_thang = 0;

            /* need to reset these too */
            x = outl->x;
            y = outl->y;

            /* re-calculate the y limits of the band */
            y0 = FIXED_ROUND((outl->hi_y - outl->lo_y) / num_bands);
            x0 = 0; /* satisfy lint */

            band_ymin = FIXED_FLOOR(outl->lo_y) + band * y0;
            r->band_ymin = band_ymin;

            if (band == num_bands - 1)
                band_ymax = FIXED_CEILING(outl->hi_y);
            else
                band_ymax = band_ymin + y0;
            r->band_ymax = band_ymax;
            r->top = band_ymax >> 16;

            /* process each piece of the outline for this band */
            for (i = 0; i < num; i++)
            {
                switch (type[i])
                {
                case FS_MOVETO:
                case FS_MOVETO + OUTLINE_CHAR:
                    x0 = *x++;
                    y0 = *y++;
                    break;
                case FS_LINETO:
                    x1 = *x++;
                    y1 = *y++;

                    /* ? no interaction with band ... skip it */
                    ymax = ymin = y0;
                    if (y1 < ymin)
                        ymin = y1;
                    else if (y1 > ymax)
                        ymax = y1;
                    if (ymin > band_ymax || ymax < band_ymin)
                    {
                        x0 = x1;
                        y0 = y1;
                        continue;
                    }

                    /* process it */
                    ok = do_line(r, x0, y0, x1, y1);
                    if (!ok)
                        goto restart;

                    x0 = x1;
                    y0 = y1;
                    break;

                case FS_QUADTO:
                    /* push 6 quad parts onto stack */
                    stack[0] = x0;
                    stack[1] = y0;
                    stack[2] = x[0];
                    stack[3] = y[0];
                    stack[4] = x2 = x[1];
                    stack[5] = y2 = y[1];
                    stack += 6;
                    x += 2;
                    y += 2;

                    while (stack > stack_base)
                    {
                        FS_FIXED Lx, Rx, Mx, Ly, Ry, My;

                        /* pop the next 6 quad parts */
                        stack -= 6;
                        y2 = stack[5];
                        x2 = stack[4];
                        y1 = stack[3];
                        x1 = stack[2];
                        y0 = stack[1];
                        x0 = stack[0];

                        /* ? no interaction with band ... skip it */
                        ymin = ymax = y0;
                        if (y1 < ymin)
                            ymin = y1;
                        else if (y1 > ymax)
                            ymax = y1;
                        if (y2 < ymin)
                            ymin = y2;
                        else if (y2 > ymax)
                            ymax = y2;
                        if (ymin > band_ymax || ymax < band_ymin)
                            continue;

                        /* bisect the quad
                        ** p0,p1,p2 ==> p0,L,M and M,R,p2
                        **
                        **           p1
                        **           o
                        **          / \
                        **         /   \
                        **        /     \
                        **       /   M   \
                        **    L o----o----o R
                        **     /           \
                        **    /             \
                        **   /               \
                        **  o                 o
                        **  p0                p2
                        */
                        Lx = (x0 + x1) >> 1;
                        Ly = (y0 + y1) >> 1;
                        Rx = (x1 + x2) >> 1;
                        Ry = (y1 + y2) >> 1;
                        Mx = (Lx + Rx) >> 1;
                        My = (Ly + Ry) >> 1;

                        {
                            /* simple err metric                 */
                            /* ? flat enough, process it         */
                            /* WAS ... err = DistanceNorm(x1-Mx, y1-My); */
                            /* but inlining it is MUCH faster    */
                            FS_FIXED dx, dy;
                            dx = x1 - Mx;
                            if (dx < 0)
                                dx = -dx;
                            dy = y1 - My;
                            if (dy < 0)
                                dy = -dy;
                            err = (dx > dy) ? dx + ((dy + dy + dy) >> 3) : dy + ((dx + dx + dx) >> 3);
                        }

                        if (err < max_err || STACK_FULL)
                        {
                            ok = do_line(r, x0, y0, x2, y2);
                            if (!ok)
                                goto restart;
                        }
                        else
                        {
                            /* push the two halves of the quad */
                            stack[0] = Mx;
                            stack[1] = My;
                            stack[2] = Rx;
                            stack[3] = Ry;
                            stack[4] = x2;
                            stack[5] = y2;
                            stack[6] = x0;
                            stack[7] = y0;
                            stack[8] = Lx;
                            stack[9] = Ly;
                            stack[10] = Mx;
                            stack[11] = My;
                            stack += 12; /* pushed 12 parts */
                        }
                    } /* end "while" */
                    x0 = x2;
                    y0 = y2;
                    break;

                case FS_CUBETO:
                {
                    FS_FIXED err1, err2, x3, y3;
                    FS_FIXED Ax, Bx, Cx, Dx, Ex, Fx, Ay, By, Cy, Dy, Ey, Fy;

                    /* push 8 cubic parts onto stack */
                    stack[0] = x0;
                    stack[1] = y0;
                    stack[2] = x[0];
                    stack[3] = y[0];
                    stack[4] = x[1];
                    stack[5] = y[1];
                    stack[6] = x3 = x[2];
                    stack[7] = y3 = y[2];
                    stack += 8;
                    x += 3;
                    y += 3;

                    while (stack > stack_base)
                    {
                        /* pop the next 8 cubic parts -- yes, order is reverse */
                        stack -= 8;
                        y3 = stack[7];
                        x3 = stack[6];
                        y2 = stack[5];
                        x2 = stack[4];
                        y1 = stack[3];
                        x1 = stack[2];
                        y0 = stack[1];
                        x0 = stack[0];

                        /* ? no interaction with band ... skip it */
                        ymin = ymax = y0;
                        if (y1 < ymin)
                            ymin = y1;
                        else if (y1 > ymax)
                            ymax = y1;
                        if (y2 < ymin)
                            ymin = y2;
                        else if (y2 > ymax)
                            ymax = y2;
                        if (y3 < ymin)
                            ymin = y3;
                        else if (y3 > ymax)
                            ymax = y3;
                        if (ymin > band_ymax || ymax < band_ymin)
                            continue;

                        /* bisect the cubic
                        ** p0,p1,p2,p3 ==> p0,A,D,F and F,E,C,p3
                        **
                        **            p1          B             p2
                        **            o-----------o-------------o
                        **           /         __-^-__           \
                        **          /      __--       --__        \
                        **         /   __-o-------o--------o__     \
                        **        /__--   D       F        E  --__  \
                        **     A o                                -- o C
                        **      /                                     \
                        **     /                                       \
                        **    /                                         \
                        **   /                                           \
                        **  o p0                                          o p3
                        **
                        */
                        Ax = (x0 + x1) >> 1;
                        Ay = (y0 + y1) >> 1;
                        Bx = (x1 + x2) >> 1;
                        By = (y1 + y2) >> 1;
                        Cx = (x2 + x3) >> 1;
                        Cy = (y2 + y3) >> 1;
                        Dx = (Ax + Bx) >> 1;
                        Dy = (Ay + By) >> 1;
                        Ex = (Bx + Cx) >> 1;
                        Ey = (By + Cy) >> 1;
                        Fx = (Dx + Ex) >> 1;
                        Fy = (Dy + Ey) >> 1;

                        /* ? flat enough, process it */
                        err1 = DistanceNorm(x1 - Dx, y1 - Dy);
                        err2 = DistanceNorm(x2 - Ex, y2 - Ey);
                        err = MAX(err1, err2);
                        if (err < max_err || STACK_FULL)
                        {
                            ok = do_line(r, x0, y0, x3, y3);
                            if (!ok)
                                goto restart;
                        }
                        else
                        {
                            /* push the two halves of the cube */
                            stack[0]  = Fx;
                            stack[1]  = Fy;
                            stack[2]  = Ex;
                            stack[3]  = Ey;
                            stack[4]  = Cx;
                            stack[5]  = Cy;
                            stack[6]  = x3;
                            stack[7]  = y3;
                            stack[8]  = x0;
                            stack[9]  = y0;
                            stack[10] = Ax;
                            stack[11] = Ay;
                            stack[12] = Dx;
                            stack[13] = Dy;
                            stack[14] = Fx;
                            stack[15] = Fy;
                            stack += 16; /* pushed 16 parts */
                        }
                    }
                    x0 = x3;
                    y0 = y3;
                    break;
                } /* case FS_CUBETO: */
                default:
                    break;
                } /* switch() */
            } /* "for (i" */

            do_band(r);
        } /* "for (band" */
        gmap = r->gmap;
        r->ref_count--; /* allow get_some_back to free RASTER if needed */

    } /* end graymap rendering section */
    /*lint +e801  Info -- Use of goto is deprecated */

#ifdef FS_STIK
    if (is_stik)
    {
        FSS_free_char(_PS_ outl);
    }
#endif

    /*
    ** Apply special effects
    */

    /* 2-bit output looks better if we convert first */
    if (gmaptype & FS_MAP_GRAYMAP2)
    {
        gmap = trim_convert(_PS_ gmap, gmaptype);
        needs_trim_convert = 0;
    }
#ifdef FS_PSEUDO_BOLD
    if ( (STATE.bold_pct || STATE.cur_typeset.tfntarray[STATE.cur_font].cfnt->bold_pct))
    {
        pixel_bold = ( (STATE.flags & FLAGS_ADD_WEIGHT) ||
                       ((STATE.cur_lfnt->fontflags & FONTFLAG_NEW_AA_ON) &&
                        (STATE.lpm <= 26) && !(STATE.flags & FLAGS_OUTLINEBOLD)));
        if ( pixel_bold && gmap )
        {
            if (gmap->bitsPerPixel == 8)
                pixelbold_graymap8(_PS_ gmap);
            else if (gmap->bitsPerPixel == 2)
                pixelbold_graymap2(_PS_ gmap);
        }
    }
#endif

    if (STATE.flags & FLAGS_EMBOSSED)
        gmap = emboss_graymap(gmap);
    else if (STATE.flags & FLAGS_ENGRAVED)
        gmap = engrave_graymap(gmap);

    /* n-pixel outline effect */
    if (STATE.flags & FLAGS_OUTLINED)
        gmap = outline_graymap(_PS_ gmap, 1, opacity);
    else if (STATE.flags & FLAGS_OUTLINED_2PIXEL)
        gmap = outline_graymap(_PS_ gmap, 2, opacity);
    else if ((STATE.flags & FLAGS_OUTLINED_FILLED) ||
             (STATE.flags & FLAGS_OUTLINED_UNFILLED))
        gmap = outline_graymap(_PS_ gmap, outline_width, opacity);
    else if (STATE.flags & FLAGS_OUTLINED_SOFT)
        gmap = soft_outline_graymap(_PS_ gmap, outline_width, opacity);

    /* soften effect */
    if ((STATE.flags & FLAGS_SOFTENED))
    {
        /* default soften filter parameters differ by size, but          */
        /* your system or personal preferences may differ ... experiment */
        if (gmap)
        {
            if (STATE.lpm < 17)
                gmap = Soften(_PS_ 1, 1, gmap);
            else if (STATE.lpm < 26)
                gmap = Soften(_PS_ 2, 2, gmap);
            else
                gmap = Soften(_PS_ 3, 3, gmap);
        }
        if (STATE.error)
        {
            FSS_free_char(_PS_ gmap);
            return 0;
        }
    }

    if (gmap && needs_trim_convert)
        gmap = trim_convert(_PS_ gmap, gmaptype);

#ifdef FS_STATS
    made_gmap++;
#endif

    return gmap;
}

#endif /* FS_GRAYMAPS */
#if defined(FS_GRAYMAPS) || defined(FS_EDGE_RENDER)
/****************************************************************/
FS_GRAYMAP *copy_graymap(_DS_ FS_GRAYMAP *p)
{
    FS_GRAYMAP *r;

    if (p == 0)
        return 0;
#ifdef FS_MEM_DBG
    STATE.memdbgid = "FS_GRAYMAP";
#endif
    r = (FS_GRAYMAP *)FSS_malloc(_PS_ p->size);
    if (r == 0)
        return 0;

    SYS_MEMCPY(r, p, p->size);

    /* this is no longer a cached object */
    r->cache_ptr = 0;

    return r;
}

#endif /* FS_GRAYMAPS or FS_EDGE_RENDER */
/****************************************************************/

FS_GRAYMAP *get_graymap(_DS_ FS_ULONG id, FS_USHORT index, FS_USHORT type,
                        FS_SHORT phasex, FS_SHORT phasey)
{
    FS_GRAYMAP *gmap = 0;
    SFNT *sfnt;
    LFNT *lfnt;
    FS_ULONG flags = STATE.flags;
#ifdef FS_CACHE_GRAYMAPS
    FS_SHORT cache_type = 0;
#endif

    STATE.error = SUCCESS;

#if defined(FS_GRAYMAPS) && defined(FS_EXTERNAL_OUTLINE)
    if (STATE.user_outline)
    {
        FS_OUTLINE *outl;

        outl = copy_outline(_PS_ STATE.user_outline,
                            0 /* source is in server memory */);
        if (!outl)
            return 0;

        /* possibly shift outline by quarter-pixel amount */
        shift_outline(outl, phasex, phasey);

        /* then make the graymap ... */
        gmap = make_graymap(_PS_ outl, type);
        if (STATE.error)
        {
            FSS_free_char(_PS_ outl);
            return 0;
        }
        FSS_free_char(_PS_ outl);

        return gmap;
    }
#endif /* FS_GRAYMAPS */

    lfnt = STATE.cur_lfnt;
    sfnt = STATE.cur_sfnt;
    if (lfnt == 0 || sfnt == 0)
        return 0;

    STATE.flags |= FLAGS_GRAYSCALE;  /* use grayscale autohinting if applicable */

#ifdef FS_CACHE_GRAYMAPS
    if (type & FS_MAP_GRAYMAP4)
        cache_type = CACHE_GRAYMAP;
    else if (type & FS_MAP_GRAYMAP8)
        cache_type = CACHE_GRAYMAP8;
    else if (type & FS_MAP_GRAYMAP2)
        cache_type = CACHE_GRAYMAP2;

    gmap = find_graymap_in_cache(_PS_ index, cache_type, phasex, phasey ); /*lint !e838 Suppress 'Previously assigned value has not been used' */
    if (gmap)
    {
        STATE.flags = flags;
        return gmap;
    }
#endif

#ifdef FS_EMBEDDED_BITMAP
    /* no? ... look for an embedded graymap */
    if ((lfnt->fnt_type == TTF_TYPE) || (lfnt->fnt_type == TTC_TYPE))
    {
        gmap = get_embedded_graymap(_PS_ sfnt, (FS_USHORT)index, type);
        if (STATE.error)
        {
            STATE.flags = flags;
            return 0;
        }
        if (gmap)
        {
            /* apply special graymap effects */
#ifdef FS_GRAYMAPS
#ifdef FS_PSEUDO_BOLD
            {
                FS_SHORT bw = sfnt->senv->bold_width;
                if (bw)
                {
                    gmap = pixelbold_embedded_graymap(_PS_ gmap);
                }
            }
#endif
            /* emboss/engrave effect */
            if (STATE.flags & FLAGS_EMBOSSED)
                gmap = emboss_graymap(gmap);
            else if (STATE.flags & FLAGS_ENGRAVED)
                gmap = engrave_graymap(gmap);

            /* n-pixel outline effect */
            if (STATE.flags & FLAGS_OUTLINED)
                gmap = outline_graymap(_PS_ gmap, 1, STATE.outline_opacity);
            else if (STATE.flags & FLAGS_OUTLINED_2PIXEL)
                gmap = outline_graymap(_PS_ gmap, 2, STATE.outline_opacity);
            else if ((STATE.flags & FLAGS_OUTLINED_FILLED) ||
                     (STATE.flags & FLAGS_OUTLINED_UNFILLED))
                gmap = outline_graymap(_PS_ gmap, STATE.outline_width,
                                       STATE.outline_opacity);
            else if (STATE.flags & FLAGS_OUTLINED_SOFT)
                gmap = soft_outline_graymap(_PS_ gmap, STATE.outline_width,
                                            STATE.outline_opacity);

            /* soften effect */
            if ((STATE.flags & FLAGS_SOFTENED))
            {
                /* default soften filter parameters differ by size, but          */
                /* your system or personal preferences may differ ... experiment */
                if (gmap)
                {
                    if (STATE.lpm < 17)
                        gmap = Soften(_PS_ 1, 1, gmap);
                    else if (STATE.lpm < 26)
                        gmap = Soften(_PS_ 2, 2, gmap);
                    else
                        gmap = Soften(_PS_ 3, 3, gmap);
                }
                if (STATE.error)
                {
                    FSS_free_char(_PS_ gmap);
                    STATE.flags = flags;
                    return 0;
                }
            }
#endif /* FS_GRAYMAPS */

#ifdef FS_CACHE_GRAYMAPS
            /* if font is disk based, cache the embedded graymap */
            /* if gmap had special effects applied, cache it as well */
            if (lfnt->path || (STATE.flags & SPECIALEFFECTS))
                save_graymap_to_cache(_PS_ index,gmap,cache_type,phasex,phasey);
#endif
        }
    }
#endif  /* FS_EMBEDDED_BITMAP */

#ifdef FS_GRAYMAPS
    /* still no? ... then we must make it from scratch */
    if (!gmap)
    {
        FS_OUTLINE *outl;

        outl = find_or_make_outline(_PS_ lfnt, sfnt, id, index);
        if (!outl)
        {
            STATE.flags = flags;
            return 0;
        }

        /* possibly shift outline by quarter-pixel amount */
        shift_outline(outl, phasex, phasey);

        /* then make the graymap ... */
        gmap = make_graymap(_PS_ outl, type);
        if (STATE.error)
        {
            FSS_free_char(_PS_ outl);
            STATE.flags = flags;
            return 0;
        }

#ifdef FS_CACHE_OUTLINES
        /* restore the cached outline to its original unshifted state */
        shift_outline(outl, -phasex, -phasey);
#endif

        FSS_free_char(_PS_ outl);

        if (!gmap)
        {
            STATE.flags = flags;
            return 0;
        }

        /* apply any vertical shift adjustment for component font */
        if (sfnt->vertical_shift)
        {
            SENV *senv = (SENV *)(sfnt->senv);

            if (senv) /* should be true always at this point */
            {
                if (senv->vanilla)
                {
                    gmap->hi_y += sfnt->vertical_shift;
                }
                else
                {
                    FIXED_VECTOR p;
                    p.x = gmap->dx;
                    p.y = gmap->dy;
                    fixed_norm(&p);  /* unit tangent vector */
                    /* adjust graymap placement by scaled normal vector */
                    gmap->lo_x += (FS_SHORT)FixMul(-p.y, sfnt->vertical_shift);
                    gmap->hi_y += (FS_SHORT)FixMul( p.x, sfnt->vertical_shift);
                }
            }
        }

#ifdef FS_CACHE_GRAYMAPS
        save_graymap_to_cache(_PS_ index, gmap, cache_type, phasex, phasey);
#endif
    }
#else  /* FS_GRAYMAPS */
    id = id; /* eliminate compiler warning */
    phasex = phasex;
    phasey = phasey;
    type = type;
    index = index;
#endif /* FS_GRAYMAPS */
    STATE.flags = flags;
    return gmap;
}

/****************************************************************/
#ifdef FS_DUMP
/* display the graymap in ASCII */
FS_VOID dump_graymap(FS_GRAYMAP *gmap)
{
    int i, j;
    FS_BYTE *bits;

    if (!gmap)
        return;

    bits = &gmap->bits[0];
    FS_PRINTF(("lo_x=%d hi_y=%d width=%d height=%d bpl=%d\n",
               gmap->lo_x, gmap->hi_y, gmap->width, gmap->height, gmap->bpl));

    FS_PRINTF(("i_dx=%d i_dy=%d dx=%12.5f dy=%12.5f\n", gmap->i_dx, gmap->i_dy,
               gmap->dx / 65536.0, gmap->dy / 65536.0));

    /* 3 digit x coords across the top */
    FS_PRINTF(("    "));
    for (i = 0; i < gmap->width; i++)
    {
        FS_PRINTF(("%d", ABS(gmap->lo_x + i) / 100));
        if (gmap->bitsPerPixel == 8)
            FS_PRINTF((" "));
    }


    FS_PRINTF(("\n    "));
    for (i = 0; i < gmap->width; i++)
    {
        FS_PRINTF(("%d", (ABS(gmap->lo_x + i) % 100) / 10));
        if (gmap->bitsPerPixel == 8)
            FS_PRINTF((" "));
    }

    FS_PRINTF(("\n    "));
    for (i = 0; i < gmap->width; i++)
    {
        FS_PRINTF(("%d", ABS(gmap->lo_x + i) % 10));
        if (gmap->bitsPerPixel == 8)
            FS_PRINTF((" "));
    }

    FS_PRINTF(("\n"));

    if (gmap->bitsPerPixel == 4 )
    {
        for (i = 0; i < gmap->height; i++)
        {
            FS_PRINTF(("%3d ", gmap->hi_y - i));
            for (j = 0; j < gmap->bpl; j++)
                FS_PRINTF(("%02x", bits[j]));
            FS_PRINTF(("\n"));
            bits += gmap->bpl;
        }
    }
    else if (gmap->bitsPerPixel == 2)
    {
        for (i = 0; i < gmap->height; i++)
        {
            FS_BYTE a, b, c, d;

            FS_PRINTF(("%3d ", gmap->hi_y - i));
            for (j = 0; j < gmap->bpl; j++)
            {
                a = 0x03 & (bits[j] >> 6);
                b = 0x03 & (bits[j] >> 4);
                c = 0x03 & (bits[j] >> 2);
                d = 0x03 & (bits[j]);

                FS_PRINTF(("%x%x%x%x", a, b, c, d));
            }
            FS_PRINTF(("\n"));
            bits += gmap->bpl;
        }
    }
    else if (gmap->bitsPerPixel == 8)
    {
        for (i = 0; i < gmap->height; i++)
        {
            FS_PRINTF(("%3d ", gmap->hi_y - i));
            for (j = 0; j < gmap->bpl; j++)
                FS_PRINTF(("%02x ", bits[j]));
            FS_PRINTF(("\n"));
            bits += gmap->bpl;
        }
    }
    else if (gmap->bitsPerPixel == 16)
    {
        for (i = 0; i < gmap->height; i++)
        {
            FS_PRINTF(("%3d ", gmap->hi_y - i));
            for (j = 0; j < gmap->bpl; j++)
                if (j % 2)
                {
                    FS_PRINTF(("%02x ", bits[j]));
                }
                else
                {
                    FS_PRINTF(("%02x", bits[j]));
                }
            FS_PRINTF(("\n"));
            bits += gmap->bpl;
        }
    }
    else if (gmap->bitsPerPixel == 32)
    {
        for (i = 0; i < gmap->height; i++)
        {
            FS_PRINTF(("%3d ", gmap->hi_y - i));
            for (j = 0; j < gmap->bpl; j++)
                if ((j + 1) % 4)
                {
                    FS_PRINTF(("%02x", bits[j]));
                }
                else
                {
                    FS_PRINTF(("%02x ", bits[j]));
                }
            FS_PRINTF(("\n"));
            bits += gmap->bpl;
        }
    }
    FS_PRINTF(("\n"));
}
#endif

/****************************************************************/
/* remove leading  trailing blank rows/columns in 8-bit graymap */
/* converts to specified bit depth                              */
#if defined(FS_GRAYMAPS) || defined(FS_EDGE_RENDER)
/*lint -e661  Warning -- Possible access of out-of-bounds pointer  */
/*lint -e662  Warning -- Possible creation of out-of-bounds pointer  */
/*lint -e669  Warning -- Possible data overrun  */
FS_GRAYMAP *trim_convert(_DS_ FS_GRAYMAP *gmap, FS_USHORT type)
{
    int fr, fc, lr, lc;
    int r, height, width, bpl, bitdepth = 8;
    FS_BYTE *p, mingray = 0;

    /* should not happen */
    if (gmap->bitsPerPixel != 8)
        return gmap;

    bpl = gmap->bpl;
    width = gmap->width;
    height = gmap->height;

    if (type & (FS_MAP_GRAYMAP4 | FS_MAP_EDGE_GRAYMAP4))
    {
        bitdepth = 4;
        mingray = 15;
        type &= (FS_MAP_GRAYMAP4 | FS_MAP_EDGE_GRAYMAP4);
    }
    else if (type & (FS_MAP_GRAYMAP2 | FS_MAP_EDGE_GRAYMAP2))
    {
        bitdepth = 2;
        mingray = 63;
        type &= (FS_MAP_GRAYMAP2 | FS_MAP_EDGE_GRAYMAP2);
    }

    /*
    ** Check whether trimming is needed
    */

    /* first non-blank row */
    p = gmap->bits;
    for (fr = 0; fr < height; fr++)
    {
        int c;
        for (c = 0; c < bpl; c++)
        {
            if (*p++ > mingray)
                break;
        }
        if (c != bpl)
            break;
    }

    /* last non-blank row */
    p = gmap->bits + bpl * (height - 1);
    for (lr = height - 1; lr >= fr; lr--)
    {
        int c;
        for (c = 0; c < bpl; c++)
        {
            if (p[c] > mingray)
                break;
        }
        if (c != bpl)
            break;
        p -= bpl;
    }

    /* recompute height */
    height = 1 + lr - fr;

    /* first non-blank column */
    for (fc = 0; fc < width; fc++)
    {
        p = gmap->bits + fr * bpl;
        for (r = 0; r < height; r++)
        {
            if (p[fc] > mingray) /*lint !e796 Suppress 'Conceivable access of out-of-bounds pointer' */
                break;
            p += bpl;            /*lint !e797 Suppress 'Possible creation of out-of-bounds pointer ' */
        }
        if (r != height)
            break;
    }

    /* last non-blank column */
    for (lc = width - 1; lc >= 0; lc--)
    {
        p = gmap->bits + fr * bpl;
        for (r = 0; r < height; r++)
        {
            if (p[lc] > mingray) /*lint !e796 Suppress 'Conceivable access of out-of-bounds pointer' */
                break;
            p += gmap->bpl;
        }
        if (r != height)
            break;
    }

    /* recompute the width */
    width = 1 + lc - fc;
    if (width < 0) width = 0; /* empty graymap, convert below */

    /* do trim and/or convert if needed */
    if ( (fr != 0 || width != gmap->bpl || height != gmap->height) || /* trim required OR    */
         (bitdepth < 8) ) /* conversion required */
    {
        /* allocate a new graymap */
        FS_GRAYMAP *new_map;
        FS_LONG size;

        /* recompute bpl for resultant graymap if bitdepth is not 8 */
        if (bitdepth == 4)
            bpl = (width + 1) >> 1;
        else if (bitdepth == 2 )
            bpl = (width + 3) >> 2;

        size = offsetof(FS_GRAYMAP, bits);
        size += height * bpl;
#ifdef FS_MEM_DBG
        STATE.memdbgid = "FS_GRAYMAP";
#endif
        new_map = (FS_GRAYMAP *)FSS_calloc(_PS_ size); /* see ITP-1975 */

        if (new_map == 0)
        {
            /* not fatal */
            STATE.error = SUCCESS;
            return gmap;
        }
        /* copy settings */
        new_map->cache_ptr = 0;
        new_map->dx = gmap->dx;
        new_map->dy = gmap->dy;
        new_map->i_dx = gmap->i_dx;
        new_map->i_dy = gmap->i_dy;
        new_map->embedded = false;
        new_map->size = size;
        new_map->lo_x = gmap->lo_x + (FS_SHORT)fc;
        new_map->hi_y = gmap->hi_y - (FS_SHORT)fr;
        new_map->width = (FS_SHORT)width;
        new_map->height = (FS_SHORT)height;
        new_map->type = type;

        if (bitdepth == 4)
        {
            FS_BYTE *gmap_p = gmap->bits + fr * gmap->bpl + fc;
            FS_BYTE *new_p = new_map->bits;
            int c, w = width & ~1;
            int odd = width & 1;
            int h = height;

            new_map->bpl = (FS_SHORT)((width + 1) >> 1);
            new_map->bitsPerPixel = 4;
            while (h--)
            {
                for (c = 0; c < w; c += 2)
                    * new_p++ = (gmap_p[c] & 0xF0) | (gmap_p[c + 1] >> 4);
                if (odd)
                    *new_p++ = (gmap_p[c] & 0xF0);
                gmap_p += gmap->bpl;
            }
        }
        else if (bitdepth == 8)
        {
            int gmap_bpl = gmap->bpl;
            FS_BYTE *gmap_p = gmap->bits + fr * gmap_bpl + fc;
            FS_BYTE *new_p = new_map->bits;

            new_map->bpl = (FS_SHORT)width;
            new_map->bitsPerPixel = 8;

            while (height--)
            {
                SYS_MEMCPY(new_p, gmap_p, width);
                new_p += width;
                gmap_p += gmap_bpl;
            }
        }
        else if (bitdepth == 2 )
        {
            FS_BYTE *gmap_p = gmap->bits + fr * gmap->bpl + fc;
            FS_BYTE *new_p = new_map->bits;
            int c, w = width & ~3;
            int h = height;
            int rem = width - w;

            new_map->bpl = (FS_SHORT)((width + 3) >> 2);
            new_map->bitsPerPixel = 2;
            new_map->type = type;

            while (h--)
            {
                for (c = 0; c < w; c += 4)
                    * new_p++ = (gmap_p[c] & 0xC0) | ((gmap_p[c + 1] >> 2) & 0x30) | ((gmap_p[c + 2] >> 4) & 0x0C) | (gmap_p[c + 3] >> 6);
                if (rem == 1)
                    *new_p++ = (gmap_p[c] & 0xC0);
                else if (rem == 2)
                    *new_p++ = (gmap_p[c] & 0xC0) | ((gmap_p[c + 1] >> 2) & 0x30);
                else if (rem == 3)
                    *new_p++ = (gmap_p[c] & 0xC0) | ((gmap_p[c + 1] >> 2) & 0x30) | ((gmap_p[c + 2] >> 4) & 0x0C);
                gmap_p += gmap->bpl;
            }
        }

        FSS_free_char(_PS_ gmap);
        return new_map;
    }
    else
        return gmap; /* untrimmed 8-bit graymap */
}
/*lint +e661  Warning -- Possible access of out-of-bounds pointer  */
/*lint +e662  Warning -- Possible creation of out-of-bounds pointer  */
/*lint +e669  Warning -- Possible data overrun  */
#endif /* FS_GRAYMAPS or FS_EDGE_RENDER */

